﻿using Microsoft.IdentityModel.Tokens;
using System;
using System.IdentityModel.Tokens.Jwt;
using System.Net;
using System.Net.Http;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading.Tasks;
using Vimt.JsonWebToken.Services.Configuration;
using Logger = VRM.Integration.Servicebus.Core.Logger;

namespace Vimt.JsonWebToken.Services.Rest
{
    /// <summary>
    /// Service Factory.
    /// </summary>
    public static class ServiceFactory
    {
        /// <summary>
        /// Encrypt Json Web Token.
        /// </summary>
        /// <param name="payload">Json Web Token Payload.</param>
        /// <returns>Encrypted Json Web Token Token.</returns>
        public static async Task<string> EncryptToken(JwtPayload payload)
        {
            try
            {
                ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
                ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

                var jwtToken = new JwtSecurityToken(new JwtHeader(GetSigningCredentials()), payload);
                var token = new JwtSecurityTokenHandler().WriteToken(jwtToken);
                
                using (var client = new HttpClient())
                {
                    client.BaseAddress = new Uri(JwtSecurityConfiguration.Current.BaseUri);
                    client.DefaultRequestHeaders.Referrer = new Uri(JwtSecurityConfiguration.Current.RefererUri);
                    var response = await client.PostAsync(JwtSecurityConfiguration.Current.EncryptUri, new StringContent(token, Encoding.UTF8, "text/plain"));

                    if (!response.IsSuccessStatusCode)
                    {
                        Logger.Instance.Error(string.Format("Failed to POST to JWT Service. StatusCode: {0}, Reason: {1}", response.StatusCode, response.ReasonPhrase));
                        throw new Exception(string.Format("Failed to POST JWT Service. Reason : {0}", response));
                    }

                    var result = await response.Content.ReadAsStringAsync();
                    Logger.Instance.Debug(string.Format("JWT POST Result: {0}", result));

                    return result;
                }
            }
            catch (Exception e)
            {
                Logger.Instance.Error(string.Format("Unexpected error occured. Failed to POST JWT Service. Reason : {0}", e.Message));
                throw new Exception(string.Format("Unexpected error occured. Failed to POST JWT Service. Reason : {0}", e.Message), e.InnerException);
            }
        }

        /// <summary>
        /// Get Signing Credentials.
        /// </summary>
        /// <returns>Signing Credentials.</returns>
        private static SigningCredentials GetSigningCredentials()
        {
            var store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);

            var certificateThumbPrint = JwtSecurityConfiguration.Current.CertificateThumbPrint;

            var cert = store.Certificates.Find(X509FindType.FindByThumbprint, certificateThumbPrint, true);

            if (cert.Count < 1)
            {
                throw new Exception(string.Format("Could not find a valid certificate with thumbprint {0}", certificateThumbPrint));
            }

            var signingCredentials = new SigningCredentials(new X509SecurityKey(cert[0]), SecurityAlgorithms.RsaSha512);
            store.Close();

            return signingCredentials;
        }
    }
}